/*
 * Decompiled with CFR 0.152.
 */
package com.skynex.mylands.service;

import com.skynex.mylands.config.ConfigManager;
import com.skynex.mylands.config.Messages;
import com.skynex.mylands.config.QuestConfig;
import com.skynex.mylands.manager.LandManager;
import com.skynex.mylands.model.Land;
import com.skynex.mylands.model.Quest;
import com.skynex.mylands.model.QuestProgress;
import com.skynex.mylands.model.QuestType;
import com.skynex.mylands.util.PluginLogger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import org.bukkit.Bukkit;
import org.bukkit.Material;
import org.bukkit.entity.Player;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class QuestService {
    private final Messages messages;
    private final ConfigManager configManager;
    private QuestConfig questConfig;
    private final Map<UUID, List<QuestProgress>> playerQuestProgress = new ConcurrentHashMap<UUID, List<QuestProgress>>();
    private final Map<UUID, List<QuestProgress>> landQuestProgress = new ConcurrentHashMap<UUID, List<QuestProgress>>();
    private final Map<QuestType, List<Quest>> availableQuests = new HashMap<QuestType, List<Quest>>();
    private final Map<UUID, String> activeQuestByPlayer = new ConcurrentHashMap<UUID, String>();
    private String questMode = "personal";
    private String rewardDistribution = "equal";
    private int minContributions = 5;

    public QuestService(Messages messages, ConfigManager configManager) {
        this.messages = messages;
        this.configManager = configManager;
        this.loadQuestsFromConfig();
    }

    public QuestService(Messages messages, QuestConfig questConfig) {
        this.messages = messages;
        this.questConfig = questConfig;
        this.configManager = null;
        this.loadQuestsFromConfig();
    }

    public void configureQuestMode(String mode, String rewardDistribution, int minContributions) {
        this.questMode = mode;
        this.rewardDistribution = rewardDistribution;
        this.minContributions = minContributions;
    }

    public boolean isCollaborativeMode() {
        return "collaborative".equals(this.questMode);
    }

    public void loadQuestsFromConfig() {
        PluginLogger.info("Starting to load quests from config...", new Object[0]);
        this.availableQuests.clear();
        if (this.questConfig == null && this.configManager != null) {
            PluginLogger.info("QuestConfig is null, loading from configManager...", new Object[0]);
            this.questConfig = new QuestConfig(this.configManager.getQuestsConfig().getValues(false));
        }
        if (this.questConfig == null) {
            PluginLogger.warn("QuestConfig is null, cannot load quests", new Object[0]);
            return;
        }
        PluginLogger.info("QuestConfig loaded successfully, parsing quests...", new Object[0]);
        Map<String, List<QuestConfig.QuestData>> allQuests = this.questConfig.getAllQuests();
        PluginLogger.info("Loading quests from config, found {} quest types", allQuests.size());
        for (Map.Entry<String, List<QuestConfig.QuestData>> entry : allQuests.entrySet()) {
            String questType = entry.getKey();
            List<QuestConfig.QuestData> questDataList = entry.getValue();
            PluginLogger.info("Processing quest type: {} with {} quests", questType, questDataList.size());
            ArrayList<Quest> quests = new ArrayList<Quest>();
            for (QuestConfig.QuestData questData : questDataList) {
                if (questData.enabled()) {
                    PluginLogger.info("Loading quest: {} ({})", questData.name(), questData.id());
                    Quest quest = new Quest(questData.id(), questData.name(), questData.description(), questData.material(), questData.objectives(), questData.reward(), questData.rewardType(), questData.dailyReset(), questData.requiredLevel(), false);
                    quests.add(quest);
                    continue;
                }
                PluginLogger.info("Skipping disabled quest: {}", questData.id());
            }
            try {
                String upperQuestType = questType.toUpperCase();
                PluginLogger.info("Converting quest type '{}' to '{}'", questType, upperQuestType);
                QuestType type = QuestType.valueOf(upperQuestType);
                this.availableQuests.put(type, quests);
                PluginLogger.info("Successfully loaded {} quests for type {}", new Object[]{quests.size(), type});
            }
            catch (IllegalArgumentException e) {
                PluginLogger.warn("Unknown quest type: {} (converted to: {})", questType, questType.toUpperCase());
                PluginLogger.warn("Available quest types: {}", Arrays.toString((Object[])QuestType.values()));
            }
        }
    }

    public QuestService() {
        this.messages = null;
        this.configManager = null;
        this.initializeQuests();
    }

    private void initializeQuests() {
        ArrayList<Quest> harvestQuests = new ArrayList<Quest>();
        harvestQuests.add(new Quest("daily_harvest", "Daily Harvest", "Harvest 100 crops daily", Material.WHEAT, List.of("Harvest 100 crops"), 50, "money", true, 1, false));
        harvestQuests.add(new Quest("green_thumb", "Green Thumb", "Harvest 500 crops to become a farming expert", Material.WHEAT, List.of("Harvest 500 crops"), 200, "money", false, 1, false));
        this.availableQuests.put(QuestType.HARVEST_CROPS, harvestQuests);
        ArrayList<Quest> miningQuests = new ArrayList<Quest>();
        miningQuests.add(new Quest("daily_digger", "Daily Digger", "Mine 200 blocks daily", Material.DIAMOND_PICKAXE, List.of("Mine 200 blocks"), 40, "money", true, 1, false));
        miningQuests.add(new Quest("deep_miner", "Deep Miner", "Mine 1000 blocks to become a mining expert", Material.DIAMOND_PICKAXE, List.of("Mine 1000 blocks"), 150, "money", false, 1, false));
        this.availableQuests.put(QuestType.BREAK_BLOCKS, miningQuests);
        ArrayList<Quest> buildingQuests = new ArrayList<Quest>();
        buildingQuests.add(new Quest("daily_builder", "Daily Builder", "Place 150 blocks daily", Material.BRICKS, List.of("Place 150 blocks"), 45, "money", true, 1, false));
        buildingQuests.add(new Quest("master_builder", "Master Builder", "Place 750 blocks to become a building expert", Material.BRICKS, List.of("Place 750 blocks"), 180, "money", false, 1, false));
        this.availableQuests.put(QuestType.PLACE_BLOCKS, buildingQuests);
        PluginLogger.info("Initialized {} quest types with {} total quests", this.availableQuests.size(), this.availableQuests.values().stream().mapToInt(List::size).sum());
    }

    @NotNull
    public List<Quest> getAvailableQuests(@NotNull UUID playerId) {
        List progress = this.playerQuestProgress.getOrDefault(playerId, new ArrayList());
        Set completedQuestIds = progress.stream().filter(QuestProgress::completed).map(QuestProgress::questId).collect(Collectors.toSet());
        ArrayList<Quest> available = new ArrayList<Quest>();
        for (List<Quest> quests : this.availableQuests.values()) {
            for (Quest quest : quests) {
                if (completedQuestIds.contains(quest.id())) continue;
                available.add(quest);
            }
        }
        return available;
    }

    @NotNull
    public List<Quest> getQuestsByType(@NotNull QuestType type) {
        return this.availableQuests.getOrDefault((Object)type, Collections.emptyList());
    }

    @NotNull
    public List<QuestProgress> getQuestProgress(@NotNull UUID playerId) {
        return this.playerQuestProgress.getOrDefault(playerId, new ArrayList());
    }

    public void updateQuestProgress(@NotNull Player player, @NotNull QuestType type, int amount) {
        if (this.isCollaborativeMode()) {
            this.updateCollaborativeQuestProgress(player, type, amount);
        } else {
            this.updatePersonalQuestProgress(player, type, amount);
        }
    }

    private void updatePersonalQuestProgress(@NotNull Player player, @NotNull QuestType type, int amount) {
        UUID playerId = player.getUniqueId();
        List progress = this.playerQuestProgress.computeIfAbsent(playerId, k -> new ArrayList());
        List<Quest> typeQuests = this.availableQuests.get((Object)type);
        if (typeQuests == null) {
            return;
        }
        String activeQuestId = this.getActiveQuest(playerId);
        for (QuestProgress questProgress : progress) {
            Quest quest;
            if (questProgress.completed() || (quest = this.getQuestById(questProgress.questId())) == null || !typeQuests.contains(quest) || !quest.id().equals(activeQuestId)) continue;
            this.updateQuestProgressInternal(player, quest, questProgress, progress, amount);
        }
        if (activeQuestId == null) {
            for (Quest quest : typeQuests) {
                boolean hasProgress = progress.stream().anyMatch(p -> p.questId().equals(quest.id()));
                if (hasProgress) continue;
                QuestProgress newProgress = QuestProgress.start(playerId, quest.id());
                progress.add(newProgress);
                PluginLogger.info("Player {} started quest: {}", player.getName(), quest.name());
                break;
            }
        }
    }

    private void updateCollaborativeQuestProgress(@NotNull Player player, @NotNull QuestType type, int amount) {
        try {
            LandManager landManager = this.getLandManager();
            if (landManager != null) {
                landManager.getLandForPlayer(player.getUniqueId()).thenAccept(landOpt -> {
                    if (landOpt.isPresent()) {
                        UUID landId = ((Land)landOpt.get()).ownerId();
                        this.updateCollaborativeQuestProgressWithLand(player, type, amount, landId);
                    } else {
                        this.updatePersonalQuestProgress(player, type, amount);
                    }
                });
                return;
            }
        }
        catch (Exception e) {
            PluginLogger.warn("Failed to get land for player {}, falling back to personal mode: {}", player.getName(), e.getMessage());
        }
        this.updatePersonalQuestProgress(player, type, amount);
    }

    private LandManager getLandManager() {
        try {
            return null;
        }
        catch (Exception e) {
            return null;
        }
    }

    public void setActiveQuest(@NotNull UUID playerId, @Nullable String questId) {
        if (questId == null) {
            this.activeQuestByPlayer.remove(playerId);
        } else {
            this.activeQuestByPlayer.put(playerId, questId);
        }
    }

    @Nullable
    public String getActiveQuest(@NotNull UUID playerId) {
        return this.activeQuestByPlayer.get(playerId);
    }

    public boolean isQuestActive(@NotNull UUID playerId, @NotNull String questId) {
        return questId.equals(this.activeQuestByPlayer.get(playerId));
    }

    private void updateCollaborativeQuestProgressWithLand(@NotNull Player player, @NotNull QuestType type, int amount, @NotNull UUID landId) {
        List landProgress = this.landQuestProgress.computeIfAbsent(landId, k -> new ArrayList());
        List<Quest> typeQuests = this.availableQuests.get((Object)type);
        if (typeQuests == null) {
            return;
        }
        for (QuestProgress questProgress : landProgress) {
            Quest quest;
            if (questProgress.completed() || (quest = this.getQuestById(questProgress.questId())) == null || !typeQuests.contains(quest)) continue;
            this.updateCollaborativeQuestProgressInternal(player, quest, questProgress, landProgress, amount, landId);
        }
    }

    private void updateQuestProgressInternal(@NotNull Player player, @NotNull Quest quest, @NotNull QuestProgress questProgress, @NotNull List<QuestProgress> progressList, int amount) {
        int oldProgress = questProgress.currentAmount();
        int newProgress = Math.min(oldProgress + amount, 100);
        QuestProgress updatedProgress = questProgress.increment(amount);
        progressList.set(progressList.indexOf(questProgress), updatedProgress);
        if (newProgress >= 100 && !questProgress.completed()) {
            QuestProgress completedProgress = updatedProgress.complete();
            progressList.set(progressList.indexOf(updatedProgress), completedProgress);
            this.giveRewards(player, quest);
            PluginLogger.info("Player {} completed quest: {}", player.getName(), quest.name());
        }
        if (oldProgress < newProgress) {
            this.notifyProgress(player, quest, newProgress);
        }
    }

    private void updateCollaborativeQuestProgressInternal(@NotNull Player player, @NotNull Quest quest, @NotNull QuestProgress questProgress, @NotNull List<QuestProgress> progressList, int amount, @NotNull UUID landId) {
        int oldProgress = questProgress.currentAmount();
        int newProgress = Math.min(oldProgress + amount, 100);
        QuestProgress updatedProgress = questProgress.increment(player.getUniqueId(), amount);
        progressList.set(progressList.indexOf(questProgress), updatedProgress);
        if (newProgress >= 100 && !questProgress.completed()) {
            QuestProgress completedProgress = updatedProgress.complete();
            progressList.set(progressList.indexOf(updatedProgress), completedProgress);
            this.distributeCollaborativeRewards(quest, completedProgress, landId);
            PluginLogger.info("Land {} completed collaborative quest: {}", landId, quest.name());
        }
        if (oldProgress < newProgress) {
            this.notifyCollaborativeProgress(player, quest, newProgress, updatedProgress.getContribution(player.getUniqueId()));
        }
    }

    private void distributeCollaborativeRewards(@NotNull Quest quest, @NotNull QuestProgress completedProgress, @NotNull UUID landId) {
        if ("equal".equals(this.rewardDistribution)) {
            this.distributeRewardsToAllLandMembers(quest, landId);
        } else if ("contributor".equals(this.rewardDistribution)) {
            List<UUID> contributors = completedProgress.getActiveContributors(this.minContributions);
            for (UUID contributorId : contributors) {
                this.giveRewardsToPlayer(contributorId, quest);
            }
        }
    }

    private void distributeRewardsToAllLandMembers(@NotNull Quest quest, @NotNull UUID landId) {
        try {
            LandManager landManager = this.getLandManager();
            if (landManager != null) {
                landManager.getLand(landId).thenAccept(landOpt -> {
                    if (landOpt.isPresent()) {
                        Land land = (Land)landOpt.get();
                        this.giveRewardsToPlayer(land.ownerId(), quest);
                        for (UUID coOwnerId : land.coOwners()) {
                            this.giveRewardsToPlayer(coOwnerId, quest);
                        }
                    }
                });
            }
        }
        catch (Exception e) {
            PluginLogger.error("Failed to distribute rewards to land members: {}", e.getMessage());
        }
    }

    private void giveRewardsToPlayer(@NotNull UUID playerId, @NotNull Quest quest) {
        try {
            Player player = Bukkit.getPlayer((UUID)playerId);
            if (player == null || !player.isOnline()) {
                PluginLogger.info("Player {} is offline, rewards will be given when they log in", playerId);
                return;
            }
            this.giveRewards(player, quest);
            if (this.messages != null) {
                player.sendMessage(this.messages.collaborativeQuestCompleted(quest.name()));
            }
        }
        catch (Exception e) {
            PluginLogger.error("Failed to give rewards to player {}: {}", playerId, e.getMessage());
        }
    }

    private void notifyProgress(@NotNull Player player, @NotNull Quest quest, int progress) {
        if (this.messages != null) {
            player.sendMessage(this.messages.questProgress(quest.name(), progress));
        } else {
            player.sendMessage("\u00a7a[Quest] \u00a7e" + quest.name() + " \u00a77progress: \u00a7f" + progress + "\u00a77/\u00a7f100");
        }
    }

    private void notifyCollaborativeProgress(@NotNull Player player, @NotNull Quest quest, int progress, int contribution) {
        if (this.messages != null) {
            player.sendMessage(this.messages.questProgress(quest.name(), progress));
            player.sendMessage("\u00a77Your contribution: \u00a7e" + contribution);
        } else {
            player.sendMessage("\u00a7a[Quest] \u00a7e" + quest.name() + " \u00a77progress: \u00a7f" + progress + "\u00a77/\u00a7f100");
            player.sendMessage("\u00a77Your contribution: \u00a7e" + contribution);
        }
    }

    public boolean startQuest(@NotNull UUID playerId, @NotNull String questId) {
        Quest quest = this.getQuestById(questId);
        if (quest == null) {
            return false;
        }
        List progress = this.playerQuestProgress.computeIfAbsent(playerId, k -> new ArrayList());
        for (QuestProgress qp : progress) {
            if (!qp.questId().equals(questId)) continue;
            return false;
        }
        QuestProgress newProgress = QuestProgress.start(playerId, questId);
        progress.add(newProgress);
        return true;
    }

    @NotNull
    public List<Quest> getDailyQuests(@NotNull UUID playerId) {
        ArrayList<Quest> allQuests = new ArrayList<Quest>();
        for (List<Quest> quests : this.availableQuests.values()) {
            allQuests.addAll(quests);
        }
        Collections.shuffle(allQuests);
        return allQuests.stream().limit(3L).toList();
    }

    @Nullable
    private Quest getQuestById(@NotNull String questId) {
        for (List<Quest> quests : this.availableQuests.values()) {
            for (Quest quest : quests) {
                if (!quest.id().equals(questId)) continue;
                return quest;
            }
        }
        return null;
    }

    private void giveRewards(@NotNull Player player, @NotNull Quest quest) {
        player.giveExp(quest.reward() * 10);
        if (this.messages != null) {
            player.sendMessage(this.messages.questCompletedTitle());
            player.sendMessage(this.messages.questCompletedMessage(quest.name()));
            player.sendMessage(this.messages.questCompletedXp(quest.reward() * 10));
            player.sendMessage(this.messages.questCompletedCoins(quest.reward()));
        } else {
            player.sendMessage("\u00a7a\u00a7l\u2726 Quest Completed! \u00a7r");
            player.sendMessage("\u00a77  \u00a7e" + quest.name());
            player.sendMessage("\u00a77  \u00a76+\u00a7f" + quest.reward() * 10 + " \u00a77XP");
            player.sendMessage("\u00a77  \u00a7e+\u00a7f" + quest.reward() + " \u00a77coins");
        }
    }

    public void resetDailyQuests(@NotNull UUID playerId) {
        List<QuestProgress> progress = this.playerQuestProgress.get(playerId);
        if (progress != null) {
            progress.removeIf(qp -> qp.completed() && this.isDailyQuest(qp.questId()));
        }
    }

    private boolean isDailyQuest(@NotNull String questId) {
        return questId.startsWith("daily_");
    }

    @NotNull
    public Map<String, Integer> getQuestStats(@NotNull UUID playerId) {
        List progress = this.playerQuestProgress.getOrDefault(playerId, new ArrayList());
        HashMap<String, Integer> stats = new HashMap<String, Integer>();
        stats.put("total", progress.size());
        stats.put("completed", (int)progress.stream().filter(QuestProgress::completed).count());
        stats.put("active", (int)progress.stream().filter(qp -> !qp.completed()).count());
        return stats;
    }
}

