/*
 * Decompiled with CFR 0.152.
 */
package com.skynex.mylands.database;

import com.skynex.mylands.libs.hikari.HikariConfig;
import com.skynex.mylands.libs.hikari.HikariDataSource;
import com.skynex.mylands.util.PluginLogger;
import java.io.File;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.concurrent.CompletableFuture;
import org.bukkit.configuration.file.FileConfiguration;
import org.bukkit.plugin.Plugin;
import org.jetbrains.annotations.NotNull;

public class DatabaseManager {
    private final Plugin plugin;
    private final String databaseType;
    private HikariDataSource dataSource;

    public DatabaseManager(@NotNull Plugin plugin, @NotNull FileConfiguration config) {
        this.plugin = plugin;
        this.databaseType = config.getString("database.type", "sqlite").toLowerCase();
    }

    public CompletableFuture<Void> initialize(@NotNull FileConfiguration config) {
        return CompletableFuture.runAsync(() -> {
            try {
                HikariConfig hikariConfig = new HikariConfig();
                switch (this.databaseType) {
                    case "sqlite": {
                        this.configureSQLite(hikariConfig, config);
                        break;
                    }
                    case "mysql": 
                    case "mariadb": {
                        this.configureMySQL(hikariConfig, config);
                        break;
                    }
                    default: {
                        throw new IllegalArgumentException("Unsupported database type: " + this.databaseType);
                    }
                }
                hikariConfig.setMaximumPoolSize(10);
                hikariConfig.setMinimumIdle(2);
                hikariConfig.setConnectionTimeout(30000L);
                hikariConfig.setIdleTimeout(600000L);
                hikariConfig.setMaxLifetime(1800000L);
                hikariConfig.setLeakDetectionThreshold(60000L);
                System.setProperty("com.skynex.mylands.libs.hikari.housekeeping.periodMs", "60000");
                this.dataSource = new HikariDataSource(hikariConfig);
                this.createTables();
                PluginLogger.info("Database initialized successfully using {}", this.databaseType);
            }
            catch (Exception e) {
                PluginLogger.error("Failed to initialize database", e);
                throw new RuntimeException("Database initialization failed", e);
            }
        });
    }

    private void configureSQLite(@NotNull HikariConfig config, @NotNull FileConfiguration pluginConfig) {
        File dataFolder = this.plugin.getDataFolder();
        if (!dataFolder.exists()) {
            dataFolder.mkdirs();
        }
        String fileName = pluginConfig.getString("database.sqlite.file", "lands.db");
        File dbFile = new File(dataFolder, fileName);
        config.setJdbcUrl("jdbc:sqlite:" + dbFile.getAbsolutePath());
        config.setDriverClassName("org.sqlite.JDBC");
        config.setConnectionTestQuery("SELECT 1");
        config.addDataSourceProperty("journal_mode", "WAL");
        config.addDataSourceProperty("synchronous", "NORMAL");
        config.addDataSourceProperty("cache_size", "10000");
        config.addDataSourceProperty("foreign_keys", "ON");
    }

    private void configureMySQL(@NotNull HikariConfig config, @NotNull FileConfiguration pluginConfig) {
        String host = pluginConfig.getString("database.mysql.host", "localhost");
        int port = pluginConfig.getInt("database.mysql.port", 3306);
        String database = pluginConfig.getString("database.mysql.database", "mylands");
        String username = pluginConfig.getString("database.mysql.username", "root");
        String password = pluginConfig.getString("database.mysql.password", "");
        config.setJdbcUrl(String.format("jdbc:mysql://%s:%d/%s", host, port, database));
        config.setUsername(username);
        config.setPassword(password);
        config.setDriverClassName("com.mysql.cj.jdbc.Driver");
        config.setRegisterMbeans(false);
        config.addDataSourceProperty("cachePrepStmts", "true");
        config.addDataSourceProperty("prepStmtCacheSize", "250");
        config.addDataSourceProperty("prepStmtCacheSqlLimit", "2048");
        config.addDataSourceProperty("useServerPrepStmts", "true");
        config.addDataSourceProperty("useLocalSessionState", "true");
        config.addDataSourceProperty("rewriteBatchedStatements", "true");
        config.addDataSourceProperty("cacheResultSetMetadata", "true");
        config.addDataSourceProperty("cacheServerConfiguration", "true");
        config.addDataSourceProperty("elideSetAutoCommits", "true");
        config.addDataSourceProperty("maintainTimeStats", "false");
    }

    private void createTables() throws SQLException {
        try (Connection conn = this.getConnection();
             Statement stmt = conn.createStatement();){
            String createTableSQL = "CREATE TABLE IF NOT EXISTS lands (\n    owner_uuid VARCHAR(36) PRIMARY KEY,\n    land_name VARCHAR(32) NOT NULL,\n    world_name VARCHAR(64) NOT NULL,\n    min_x DOUBLE NOT NULL,\n    min_y DOUBLE NOT NULL,\n    min_z DOUBLE NOT NULL,\n    max_x DOUBLE NOT NULL,\n    max_y DOUBLE NOT NULL,\n    max_z DOUBLE NOT NULL,\n    level INT NOT NULL DEFAULT 1,\n    co_owners TEXT,\n    home_x DOUBLE,\n    home_y DOUBLE,\n    home_z DOUBLE,\n    home_world VARCHAR(64),\n    visits_open BOOLEAN NOT NULL DEFAULT TRUE,\n    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP\n)\n";
            stmt.execute(createTableSQL);
            if (this.databaseType.equals("mysql") || this.databaseType.equals("mariadb")) {
                stmt.execute("CREATE INDEX IF NOT EXISTS idx_world_name ON lands(world_name)");
            }
            PluginLogger.info("Database tables created/verified successfully", new Object[0]);
        }
    }

    @NotNull
    public Connection getConnection() throws SQLException {
        if (this.dataSource == null || this.dataSource.isClosed()) {
            throw new SQLException("DataSource is not initialized or has been closed");
        }
        return this.dataSource.getConnection();
    }

    public <T> CompletableFuture<T> executeAsync(@NotNull DatabaseTask<T> task) {
        return CompletableFuture.supplyAsync(() -> {
            Object t2;
            block9: {
                Connection conn = this.getConnection();
                try {
                    t2 = task.execute(conn);
                    if (conn == null) break block9;
                }
                catch (Throwable t$) {
                    try {
                        if (conn != null) {
                            try {
                                conn.close();
                            }
                            catch (Throwable x2) {
                                t$.addSuppressed(x2);
                            }
                        }
                        throw t$;
                    }
                    catch (SQLException e) {
                        PluginLogger.error("Database SQL error during async execution", e);
                        throw new RuntimeException("Database operation failed", e);
                    }
                    catch (Exception e) {
                        PluginLogger.error("Unexpected database error during async execution", e);
                        throw new RuntimeException("Unexpected database error", e);
                    }
                }
                conn.close();
            }
            return t2;
        });
    }

    public void shutdown() {
        if (this.dataSource != null && !this.dataSource.isClosed()) {
            this.dataSource.close();
            PluginLogger.info("Database connection pool closed", new Object[0]);
        }
    }

    public boolean isConnected() {
        return this.dataSource != null && !this.dataSource.isClosed();
    }

    @NotNull
    public String getDatabaseType() {
        return this.databaseType;
    }

    @FunctionalInterface
    public static interface DatabaseTask<T> {
        public T execute(Connection var1) throws SQLException;
    }
}

